function [sc,res] = scale_grw(s,tar,A0,L,ta,M,N,M0)
%--------------------------------------------------------------------------
% scale_grw.m        A scale factor is multiplied to the Cholesky factor of
%                    the variance matrix of the jumping distribution. It
%                    regulates the spread of the jumping distribution.
%                    A scale factor greater than 1 could lead to faster
%                    convergence at the price of lower acceptance
%                    probabilities. A scale factor less than 1 could lead
%                    to slower convergence, however the acceptance
%                    probability would be higher.
%                    For a Gaussian random walk jumping distribution, the
%                    recommended acceptance probability is around 25% if
%                    the number of parameters is 5 or more, and 44% in the
%                    single parameter case. 
%                 
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% s(ms,1)            Potential values of scale factor.  
% tar                Function handle - takes as input the vector of alpha
%                    and log phi, and returns the log of the target 
%                    distribution.
% A0(p,N)            Starting point of MCMC algorithm. 
% L(p,p)             Cholesky factor of variance matrix of alpha and log
%                    phi.
% ta                 Required acceptance probability. 
% M                  Length of MCMC chains.
% N                  Number of parallel chains. 
% M0                 Burn-in length for MCMC. 
% Outputs:
% sc                 Suggested scale value which leads to acceptance
%                    probabilities of around taccept
% res(ms,1)          Residuals from linear fit of log of acceptance
%                    probabilities and scale.
%--------------------------------------------------------------------------
% [sc,res] = scale_grw(s,tar,A0,L,ta,M,N,M0)                
%--------------------------------------------------------------------------
% Number of potential scale values 
  ms = length(s);

% Log of the mean of the acceptance probability for each scale
  pa = zeros(ms,1);
  for i = 1:ms
  jump = @(B)jump_grw(B,s(i)*L);
  [~,aP,~,~,~,~] = mcmcmh(tar,jump,M,N,M0,50,A0);
  pa(i) = log(mean(aP));
  end 

% Fitting a linear model with independent variable log (acceptance) and
% dependent variable scale 
  C = [ones(ms,1) pa];
  bhat = C\s;

% Evaluating the scale for which the acceptance probability is taccept
% according to the model 
  pred = [1 log(ta)];
  sc = pred*bhat;

% Residuals from the fit 
  res = s - C*bhat;
%--------------------------------------------------------------------------
% End of scale_grw.m
%--------------------------------------------------------------------------
end 