function [A,Rhat,Neff,IS,IA] = mcm2mcmc(A0,D,M0,P00)
% -------------------------------------------------------------------------
% MCM2MCMC.M       MCMC measurement uncertainty evaluation for response
%                  models of the form
%
%                  eta = phi(alpha,beta)
%
%                  based on a GUMS1/S2 MCM sample.
% -------------------------------------------------------------------------
% A B Forbes, K Jagan, I M Smith, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Design documentation 
% [1] A B Forbes, An MCMC algorithm based on GUM Supplement 1 for 
% uncertainty evaluation, Measurement, 2012, 
% DOI:10.1016/j.measurement.2012.01.018
%
% [2] A B Forbes et al, User manual for MCM2MCMC: Software implementing a
% Markov chain Monte Carlo algorithm for uncertainty evaluation based on 
% Supplement 1 to the GUM, NPL Report NPL Report MS/20, 2018
% -------------------------------------------------------------------------
% Input parameters
% A0(M,N,l)        An array which stores MCM samples of size M x N 
%                  for each of l parameters.
% D(M,N)           Absolute value of the determinant of the Jacobian 
%                  matrix partial phi_i/ partial alpha_j evaluated at
%                  (a0qr,bqr).
% M0               Length of burn-in period with M > M0 >= 0.
% <Optional ...
% P00(M,N)         Prior p(alpha) for alpha evaluated at a0qr.
%                  Default: P00(q,r) = 1.
% ...>
%
% Output parameters
% A(M,N,l)         An array which stores N Markov chains of length M
%                  determined from A0, D and P00.
% Rhat(l,1)        Convergence index for each parameter.
% Neff(l,1)        Effective number of independent draws for each
%                  parameter.
% IS(M,N)          Selection indices used to create the Markov chains
%                  from the MCM samples.
% IA(M,N)          Acceptance indices.
% -------------------------------------------------------------------------
% [A,Rhat,Neff,IS,IA] = mcm2mcmc(A0,D,M0,P00)
% -------------------------------------------------------------------------
  l = size(A0,3);
% 
% Initialisation
  Rhat = zeros(l,1);
  Neff = zeros(l,1);
  [M,N] = size(D);
  A = zeros(M,N,l);
% 
% Assign default value to P00 if required.
  if nargin == 3
      P00 = ones(M,N);
  end % if nargin 
%
% Check feasibility of P00. 
  p00q = zeros(M,1);
  for q = 1:M
      p00q(q,:) = min(P00(q,:));
  end % for q 
%
  if p00q(1) <=0
      error('P00(1,r) must be strictly positive for all r')
  end % if p00q 
%
  if p00q(2:M) < zeros(M-1,1)
      error('P00(1,r) must be non negative for all r')
  end % if p00q 
%
% Calculate selection and acceptance indices. P00 is associated with the
% target distribution, D with the approximating/proposal distribution.
  [IS,IA] = mhicI(P00,D);
%
% Run the chains for each parameter.
  for j = 1:l
      A(:,:,j) = mhicia2a(A0(:,:,j),IS);
    % Calculate convergence indices.
      [Rhat(j),Neff(j)] = mcmcci(A(:,:,j),M0);   
  end % for j 
% -------------------------------------------------------------------------
% End of mcm2mcmc.m
% -------------------------------------------------------------------------