function [sc,res] = scale_trw(s,tar,A0,R,nu,ta,M,N,M0)
%--------------------------------------------------------------------------
% scale_trw.m        A scale factor is multiplied to the Cholesky factor of
%                    the variance matrix of the jumping distribution. It
%                    regulates the spread of the jumping distribution.
%                    A scale factor greater than 1 could lead to faster
%                    convergence at the price of lower acceptance
%                    probabilities. A scale factor less than 1 could lead
%                    to slower convergence, however the acceptance
%                    probability would be higher.
%                    For a Gaussian random walk jumping distribution, the
%                    recommended acceptance probability is around 25%
%                    and the scale factor is adjusted to yield this
%                    level of acceptance.
%                    We could also use this level of acceptance for Student
%                    t random walk algorithms.
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% s(ms,1)            Potential values of scale factor.  
% tar                Function handle - takes as input the vector of alpha
%                    and returns the log of the target distribution.                  
% A0(p,N)            Starting point of MCMC algorithm. 
% R(p,p)             QR factorisation of Jacobian matrix.
% nu                 Degrees of freedom for the t jumping distribution.
% taccept            Required aceptance percentage. 
% M                  Length of MCMC chains.
% N                  Number of parallel chains. 
% M0                 Burn-in length for MCMC. 
% Outputs:
% sopt               Suggested scale value which leads to acceptance
%                    probabilities of around taccept.
% res                Residuals from linear fit of log of acceptance
%                    probabilities and scale.
%--------------------------------------------------------------------------
% [sc,res] = scale_trw(s,tar,A0,R,nu,ta,M,N,M0)                        
%--------------------------------------------------------------------------
% Number of potential scale values 
  ms = length(s);

% Log of the mean of the acceptance probability for each scale
  pa = zeros(ms,1);
  for i = 1:ms
  jump = @(B)jump_trw(B,R/s(i),nu);
  [~,aP,~,~,~,~] = mcmcmh(tar,jump,M,N,M0,50,A0);
  pa(i) = log(mean(aP));
  end 

% Fitting a linear model with independent variable log(acceptance) and
% dependent variable scale 
  C = [ones(ms,1) pa];
  fit = C\s;

% Evaluating the scale for which the acceptance probability is 25%
% according to the model 
  pred = [1 log(ta)];
  sc = pred*fit;
  
% Residuals from the fit 
  res = s - C*fit;  
%--------------------------------------------------------------------------
% End of scale_trw.m
%--------------------------------------------------------------------------
end 