function [S,aP,Rh,Ne,AA,IAA,sc] = nllsmh_grw(ahat,fun,m0,s0,M,N,M0,Q,I)
%--------------------------------------------------------------------------
% nllsmh_grw.m       Metropolis-Hastings MCMC sampling from the posterior
%                    distribution of alpha and log phi.
%                    The data arises according to a non-linear function of
%                    some parameters alpha and is subject to Gaussian noise
%                    with precision phi.
%                    Alpha has non-informative prior and Phi has gamma prior
%                    characterised by a degree of belief term, m0 and
%                    standard deviation term s0.
%                    $\phi \sim \rm{G}(m_0/2,m_0 s_0^2/2)$
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Input parameters:
% ahat(n,1)          Non-linear least squares estimates of model
%                    parameters.
% fun                Function handle that takes as input alpha values
%                    and returns the non linear least squares function, f,
%                    and the matrix of derivatives of f with respect to
%                    each input, J.
% m0                 Prior degrees of freedom.
% s0                 Prior estimate of standard deviation.
% M                  Chain length for MCMC.
% N                  Number of parallel chains.
% M0                 Burn-in length for MCMC.
% Q(nQ,1)            Quantiles to be evaluated. Ranges from 0 (minimum of
%                    sample) to 100 (maximum of sample). 50 represents
%                    the median value.
% I(true or false)   Logical variable, if I = true, then a scale factor for
%                    the jumping distribution has to be input.
%                    If I = false, then the scale is automatically determined
%                    such that the acceptance probability is around 25%.
%           Note:    A scale factor is multiplied to the Cholesky factor of
%                    the variance matrix of the jumping distribution. It
%                    regulates the spread of the jumping distribution.
%                    A scale factor greater than 1 could lead to faster
%                    convergence at the price of lower acceptance
%                    probabilities. A scale factor less than 1 could lead
%                    to slower convergence, however the acceptance
%                    probability would be higher.
%
% Output parameters:
% S(2+nQ,n)          Summary of A
%                    [ mean
%                      standard deviation
%                      percentile limits ]
%                    where the percentile limits are given by Q.
% aP(N,1)            Acceptance percentages for AA calculated for each
%                    parallel chain.
% Rh(n,1)            Estimate of convergence. Theoretically Rh >= 1, and
%                    the closer to 1, the more evidence of convergence. For
%                    example, if Rh < 1.05, then the expected improvement
%                    in the estimate of the standard deviation will be no
%                    bigger than 5 %.
% Ne(n,1)            Estimate of the number of effective number of
%                    independent draws.
%           Note:    S, Rh and Ne are calculated post-burn, ie from sample
%                    M0+1 onward.                   
% AA(M,N,n)          Array storing the chains: A(i,j,k) is the
%                    kth element of the parameter vector stored as the
%                    ith member of the jth chain.
%                    AA(1,j,:) = A0(:,j).
% IAA(M,N)           Acceptance indices.
%                    IAA(i,j) = 1 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    accepted so that AA(i,j,:) = as.
%                    IAA(i,j) = 0 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    rejected so that AA(i,j,:) = AA(i-1,j,:), i > 1.
% sc                 Scale factor used for the computation. 
%--------------------------------------------------------------------------
% [S,aP,Rh,Ne,AA,IAA,sc] = nllsmh_grw(ahat,fun,m0,s0,M,N,M0,Q,I)
%--------------------------------------------------------------------------

% Estimates of log phi as a starting point to find MAP estimate
  lamhat = log(1/s0^2);
  a0 = [ahat;lamhat];
  p = length(a0);

% Setting up target distribution
  tar = @(alp)tar_grw(alp,fun,m0,s0);

% Evaluating map estimates of parameters
  map = @(alp)eg_grw(alp,fun,tar,m0,s0);
  options = optimoptions('fminunc','GradObj','on');
  ac = fminunc(map,a0,options);

% Finite difference approximation of Hessian at MAP estimate
  del = 1e-8;
  H = zeros(p,p);
  for j = 1:p
      aval = ac;
      aval(j) = aval(j) + del;
      [~, gj1] = map(aval);
      aval = ac;
      aval(j) = aval(j) - del;
      [~, gj2] = map(aval);
      H(:,j) = (gj1(:)-gj2(:))/(2*del);
  end
  H = (H+H')/2;

% Estimate of variance matrix of alpha and log phi and its Cholesky factor
  V = inv(H);
  L = chol(V)';

% Starting points for MCMC
  ALP0 = ac*ones(1,N) + L*randn(p,N);

% Scale factor determination
  if I == true
      sc = input('scale factor');
  elseif I == false
      k = 2.4/sqrt(p);
      s = [k/8;k/4;k/2;k;2*k;4*k;8*k];
      ta = input('required acceptance probability (0-100)');
      sc = scale_grw(s,tar,ALP0(:,1:5),L,ta,500,5,50);
  end

% Gaussian random walk jumping distribution
  jump = @(alp)jump_grw(alp,sc*L);

% MCMC samples, summary and convergence information
  [S,aP,Rh,Ne,AA,IAA] = mcmcmh(tar,jump,M,N,M0,Q,ALP0);
%--------------------------------------------------------------------------
% End of nllsmh_grw.m
%--------------------------------------------------------------------------
end