function [S,aP,Rh,Ne,AA,IAA] = nllsmh_ngic(ahat,fun,m0,s0,sc,M,N,M0,Q)
% -------------------------------------------------------------------------
% nllsmhic_ngic.m    Metropolis-Hastings MCMC sampling from the posterior
%                    distribution of alpha and phi using an independence
%                    chain algorithm. 
%                    Data arises according to a non-linear function of
%                    some parameters alpha and is subject to Gaussian 
%                    noise with precision phi.
%                    Alpha has non-informative prior and Phi a gamma 
%                    prior characterised by a degree of belief term, 
%                    m0 and standard deviation term s0.
%                    $\phi \sim \rm{G}(m_0/2,m_0 s_0^2/2)$
% -------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Inputs:
% ahat(n,1)          Non-linear least squares estimates of model
%                    parameters.
% fun                Function handle that takes as input alpha values.
%                    and returns the non linear least squares function, f,
%                    and the matrix of derivatives of f with respect to
%                    each input, J.
% m0                 Prior degrees of freedom.
% s0                 Prior estimate of standard deviation.
% sc                 Scale factor for the jumping distribution. A scale 
%                    factor greater than 1 makes the jumping distribution
%                    more dispersed. 
% M                  Chain length for MCMC.
% N                  Number of parallel chains.
% M0                 Burn-in length for MCMC.
% Q(nQ,1)            Quantiles to be evaluated. Ranges from 0 (minimum of
%                    sample) to 100 (maximum of sample). 50 represents
%                    the median value.
% Outputs:
% S(2+nQ,n)          Summary of A
%                    [ mean
%                      standard deviation
%                      percentile limits ]
%                    where the percentile limits are given by Q.
% aP(N,1)            Acceptance percentages for AA calculated for each
%                    parallel chain
% Rh(n,1)            Estimate of convergence. Theoretically Rh >= 1, and
%                    the closer to 1, the more evidence of convergence. For
%                    example, if Rh < 1.05, then the expected improvement
%                    in the estimate of the standard deviation will be no
%                    bigger than 5 %.
% Ne(n,1)            Estimate of the number of effective number of
%                    independent draws.
%          Note:     S, Rh and Ne are calculated post-burn, ie from sample
%                    M0+1 onward.
% AA(M,N,n)          Array storing the chains: A(i,j,k) is the
%                    kth element of the parameter vector stored as the
%                    ith member of the jth chain.
%                    AA(1,j,:) = A0(:,j).
% IAA(M,N)           Acceptance indices.
%                    IAA(i,j) = 1 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    accepted so that AA(i,j,:) = as.
%                    IAA(i,j) = 0 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    rejected so that AA(i,j,:) = AA(i-1,j,:), i > 1.
% -------------------------------------------------------------------------
% [S,aP,Rh,Ne,AA,IAA] = nllsmhic_ngic(ahat,fun,m0,s0,sc,M,N,M0,Q)
% -------------------------------------------------------------------------
% Function and Jacobian evaluated at ahat
  [f,J] = fun(ahat);
 
% % Length of data and number of parameters 
  [m,n] = size(J);
 
% Evaluating s2bar 
  sighat = norm(f)/sqrt(m-n);
  nu = m + m0 - n;
  s2bar = (m0*s0^2 + (m-n)*sighat^2)/nu;

% QR factorisation of Jacobian matrix 
  [~,R] = qr(J,0);

% Initial values 
  A0 = jump_ngic(ahat,nu,s2bar,R,N);

% Target and jumping 
  target = @(ap)tar_ngic(ap,fun,m0,s0);
  jump = @()jump_ngic(ahat,nu/sc,s2bar,R/sc,N);
  
% Calling MCMCMH 
  [S,aP,Rh,Ne,AA,IAA] = mcmcmhic(target,jump,M,N,M0,Q,A0);
% -------------------------------------------------------------------------
% End of nllsmhic_ngic.m
% -------------------------------------------------------------------------
end 
