function [S, aP, Rh, Ne, AA, IAA] = mcmcmh(target, jump, M, N, M0, Q, A0)
% -------------------------------------------------------------------------
% mcmcmh.m          Metrolopolis-Hasting MCMC algorithm generating
%                   N chains of length M for a parameter vector A of
%                   length n.
% -------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Inputs:
% target            Function handle. This function takes as input n x N
%                   array A(n,N) and returns a 1 x N vector logp where
%                   logp(j) is the logarithm of target distribution
%                   evaluated at A(:,j), up to an additive constant.
% jump              Function handle. This function takes as input a
%                   n x N array A(n,N), current states, and returns
%                   As(n,N), proposed states, and a 1 x N vector dlogq.
%                   dlogq(j) is the difference between the logarithm of
%                   the jumping distribution associated with moving
%                   from A(:,j) to As(:,j) and that associated with moving
%                   from As(:,j) to A(:,j), up to an additive constant
%                   logJ(a|as) - logJ(as|a).
% M                 Length of the chains.
% N                 Number of chains.
% M0                Burn in period.
% Q(nQ,1)           Percentiles 0 <= Q(k) <= 100.
% A0(n,N)           Array of feasible starting points: the target
%                   distribution evaluated at A0(:,j) is strictly positive.
%
% Outputs:
% S(2+nQ,n)         Summary of A
%                   [ mean
%                     standard deviation
%                     percentile limits ]
%                   where the percentile limits are given by Q.
% aP(N,1)           Acceptance percentages for AA calculated for each
%                   chain.
% Rh(n,1)           Estimate of convergence. Theoretically Rh >= 1, and
%                   the closer to 1, the more evidence of convergence.
% Ne(n,1)           Estimate of the number of effective number of
%                   independent draws.
%           Note:   S, Rh and Ne are calculated post-burn, ie from sample
%                   M0+1 onwards.
% AA(M,N,n)         Array storing the chains: A(i,j,k) is the
%                   kth element of the parameter vector stored as the
%                   ith member of the jth chain.
%                   AA(1,j,:) = A0(:,j).
% IAA(M,N)          Acceptance indices.
%                   IAA(i,j) = 1 means that the proposal as(n,1)
%                   generated at the ith step of the jth chain was
%                   accepted so that AA(i,j,:) = as.
%                   IAA(i,j) = 0 means that the proposal as(n,1)
%                   generated at the ith step of the jth chain was
%                   rejected so that AA(i,j,:) = AA(i-1,j,:), i > 1.
%                   The first set of proposal coincide with A0 are all
%                   accepted, so IAA(1,j) = 1.
% -------------------------------------------------------------------------
% [S, aP, Rh, Ne, AA, IAA] = mcmcmh(target, jump, M, N, M0, Q, A0)
% -------------------------------------------------------------------------

% number of parameters for which samples are to be drawn
  n = size(A0, 1);

% number of percentiles to be evaluated
  nQ = length(Q);

% initialising output arrays
  AA = zeros(M, N, n);
  IAA = zeros(M, N);
  Rh = zeros(n, 1);
  Ne = zeros(n, 1);
  S = zeros(2 + nQ, n);

% starting values of the sample and associated log of target density
  aq = A0;
  lq = target(aq);

% starting values must be feasible for each chain
  Id = lq > -Inf;
  if sum(Id) < N
      error('Initial values must be feasible for all chains')
  end

% run the chains in parallel
  for q = 1:M
    % draw from the jumping distribution and calculate
    % d = log P0(aq|as) - log P0(as|aq)
      [as, d] = jump(aq);
    % log of the target density for the new draw as
      ls = target(as);
    % Metropolis-Hastings acceptance ratio
      rq = exp(ls - lq + d);
    % draws from the uniform distribution for acceptance rule
      uq = rand(1, N);
      ind = uq < rq;
    % updating parameter values and evaluating acceptance indices
      aq(:, ind) = as(:, ind);
      lq(ind) = ls(ind);
      IAA(q, ind) = 1;
    % Metropolis hastings sample
      AA(q, :, :) = aq';
  end

% acceptance probabilities for each chain
  aP = 100*sum(IAA)'/M;

% convergence and summary statistics for each of the n parameters
  for j = 1:n
    % test convergence
      [Rh(j), Ne(j)] = mcmcci(squeeze(AA(:,:,j)), M0);
    % provide summary information
      [S(1,j), S(2,j), S(3:2+nQ,j)] = ...
          mcsums(squeeze(AA(:,:,j)), M0, Q);
  end

% get rid of a singleton dimension, if required
  if n == 1
      AA = squeeze(AA);
  end
% -------------------------------------------------------------------------
% End of mcmcmh.m
% -------------------------------------------------------------------------
end