function [Rhat, Neff] = mcmcci(A, M0)
% -------------------------------------------------------------------------
% mcmcci.m          MCMC convergence indices for multiple chains.
% -------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Inputs:
% A(M, N)           Chain samples, N chains of length M.
% M0                Length of burn-in, M > M0 >= 0.
%
% Outputs:
% Rhat              Convergence index. Rhat is expected to be greater than
%                   1. The closer Rhat is to 1, the better the convergence.
% Neff              Estimate of the effective number of independent draws
%                   Neff is expected to be less than (M-M0)*N.
%                   Note: If the calculated value of Rhat is < 1, then Rhat
%                   is set to 1 and Neff set to (M-M0)*N, their
%                   limit values.
%                   Note: If N = 1 or M0 > M-2, Rhat = 0; Neff = 0
% -------------------------------------------------------------------------
% [Rhat, Neff] = mcmcci(A, M0)
% -------------------------------------------------------------------------

  [M,N] = size(A);

% Initialisation
  Rhat = 0;
  Neff = 0;

% The convergence statistics can only be evaluated if there are multiple
% chains and the chain length is greater than the burn in length
  if N > 1 && M > M0+1
    
      Mt = M - M0;
    
      ad = mean(A(M0+1:M,:))';
      add = mean(ad);
    
    % Within group standard deviation
      ss = std(A(M0+1:M,:))';
    
    % Between groups variance.
      B = Mt*(ad-add)'*(ad-add)/(N-1);
    
    % Within groups variance
      W = ss'*ss/N;
    
    % V plus
      Vp = (Mt-1)*W/Mt + B/Mt;
    
    % Convergence statistic, effective number of independent samples
      Rhat = sqrt(Vp/W);
      Neff = Mt*N*Vp/B;
    
      Rhat = max(Rhat,1);
      Neff = min(Neff,Mt*N);
  end
% -------------------------------------------------------------------------
% End of mcmcci.m
% -------------------------------------------------------------------------
end