function [S,aP,Rh,Ne,AA,IAA] = nllsmh_tic(ahat,fun,m0,s0,sc,M,N,M0,Q)                                        
%--------------------------------------------------------------------------
% nllsmh_tic.m       Metropolis-Hastings MCMC sampling from the posterior
%                    distribution of alpha.
%                    The data arises according to a non-linear function of
%                    some parameters alpha and is subject to Gaussian noise
%                    with precision phi.
%                    Alpha has non-informative prior and Phi a gamma prior
%                    characterised by a degree of belief term, m0 and
%                    standard deviation term s0.
%                    $\phi \sim \rm{G}(m_0/2,m_0 s_0^2/2)$. 
%                    Samples from marginal posterior distribution of alpha.
%                    t independence chain jumping distribution used.          
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Input parameters:
% ahat(n,1)          Non-linear least squares estimates of model
%                    parameters.
% fun                Function handle that takes as input alpha values
%                    and returns the non linear least squares function, f,
%                    and the matrix of derivatives of f with respect to
%                    each input, J.
% m0                 Prior degrees of freedom.
% s0                 Prior estimate of standard deviation.
% sc                 Scaling factor of the spread of the jumping
%                    distribution.  
% M                  Chain length for MCMC.
% N                  Number of parallel chains.
% M0                 Burn-in length for MCMC.
% Q(nQ,1)            Quantiles to be evaluated. Ranges from 0(minimum of
%                    sample) to 100(maximum of sample). 50 represents
%                    Median value.
% Output parameters:
% S(2+nQ,n)          Summary of A
%                    [ mean
%                      standard deviation
%                      percentile limits ]
%                    where the percentile limits are given by Q.
% aP(N,1)            Acceptance percentages for AA calculated for each
%                    parallel chain.
% Rh(n,1)            Estimate of convergence. Theoretically Rh >= 1, and
%                    the closer to 1, the more evidence of convergence. For
%                    example, if Rh < 1.05, then the expected improvement
%                    in the estimate of the standard deviation will be no
%                    bigger than 5 %.
% Ne(n,1)            Estimate of the number of effective number of
%                    independent draws.
%           Note:    S, Rh and Ne are calculated post-burn, ie from sample
%                    M0+1 onward.
% AA(M,N,n)          Array storing the chains: A(i,j,k) is the
%                    kth element of the parameter vector stored as the
%                    ith member of the jth chain.
%                    AA(1,j,:) = A0(:,j).
% IAA(M,N)           Acceptance indices.
%                    IAA(i,j) = 1 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    accepted so that AA(i,j,:) = as.
%                    IAA(i,j) = 0 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    rejected so that AA(i,j,:) = AA(i-1,j,:), i > 1.
%--------------------------------------------------------------------------
% [S,aP,Rh,Ne,AA,IAA] = nllsmh_tic(ahat,fun,m0,s0,sc,M,N,M0,Q)  
%--------------------------------------------------------------------------

% Non-linear function and Jacobian evaluated at ahat 
  [f,J] = fun(ahat);

% Length of data and number of parameters 
  [m,n] = size(J);
 
% Estimate of standard deviation of data 
  sighat = norm(f)/sqrt(m-n);
 
% Degrees of freedom of t distribution 
  nu = m0 + m - n;
 
% Weighted variance 
  s2bar = (m0*s0^2 + (m-n)*sighat^2)/nu;
  sbar = sqrt(s2bar);

% QR factorisation of Jacobian
  [~,R] = qr(J,0);
  Rr = R/sbar;

% Target and jumping distributions 
  tar = @(a)tar_trw(a,fun,m0,s0);
  jump = @()jump_tic(ahat,Rr/sc,nu,N);
  A0 = jump_tic(ahat,Rr,nu,N);
 
% MCMC 
  [S,aP,Rh,Ne,AA,IAA] = mcmcmhic(tar,jump,M,N,M0,Q,A0);
end 
%--------------------------------------------------------------------------
% End of nllsmh_tic.m
%--------------------------------------------------------------------------