function [Rhat,Neff,Abar,stdA] = mcmcci(A,M0)
% -------------------------------------------------------------------------
% MCMCCI.M          MCMC convergence indices for multiple chains.
% -------------------------------------------------------------------------
% A B Forbes, K Jagan, I M Smith, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Design documentation 
% [1] A B Forbes, An MCMC algorithm based on GUM Supplement 1 for 
% uncertainty evaluation, Measurement, 2012, 
% DOI:10.1016/j.measurement.2012.01.018
%
% [2] A B Forbes et al, User manual for MCM2MCMC: Software implementing a
% Markov chain Monte Carlo algorithm for uncertainty evaluation based on 
% Supplement 1 to the GUM, NPL Report MS/20, 2018
% -------------------------------------------------------------------------
% Input parameters
% A(M,N)            Chain samples, N chains of length M.
% M0                Length of burn-in, M > M0 >= 0.
%
% Output parameters
% Rhat              Convergence index. Rhat is expected to be greater than
%                   1. The closer Rhat is to 1, the better the convergence.
%
% Neff              Estimate of the effective number of independent draws.
%                   Neff is expected to be less than (M-M0)*N.
%                   Note: If the calculated value of Rhat is < 1, then Rhat 
%                   is set to 1 and Neff set to (M-M0)*N, their 
%                   limit values.
%                   Note: If N = 1 or M0 > M-2, Rhat = 0; Neff = 0. 
% Abar(M,1)         Row-wise mean.
% stdA(M,1)         Row-wise standard deviation.
% -------------------------------------------------------------------------
% [Rhat,Neff,Abar,stdA] = mcmcci(A,M0)
% -------------------------------------------------------------------------
  [M,N] = size(A);
% 
% Initialisation.
  Rhat = 0; Neff = 0;
%
  if N > 1 && M > M0+1
%    
    Mt = M - M0;
    ad = mean(A(M0+1:M,:))'; % a dot
    add = mean(ad); % a dot dot
    ss = std(A(M0+1:M,:))';
%    
%   Between groups variance.
    B = Mt*(ad-add)'*(ad-add)/(N-1);
%    
%   Within groups variance.
    W = ss'*ss/N;
%    
    Vp = (Mt-1)*W/Mt + B/Mt; % V plus
%    
    Rhat = sqrt(Vp/W);
    Neff = Mt*N*Vp/B;
%    
    Rhat = max(Rhat,1);
    Neff = min(Neff,Mt*N);
%    
  end % if N, M 
%
% Calculate means and standard deviation for each row.
  Abar = mean(A,2);
  stdA = std(A,0,2);
% -------------------------------------------------------------------------
% End of mcmcci.m
% -------------------------------------------------------------------------