function [S,aP,Rh,Ne,AA,IAA,sc] = nllsmh_trw(ahat,fun,m0,s0,M,N,M0,Q,I)
%--------------------------------------------------------------------------
% nllsmh_trw.m       Metropolis-Hastings MCMC sampling from the posterior
%                    distribution of alpha.
%                    The data arises according to a non-linear function of
%                    some parameters alpha and is subject to Gaussian noise
%                    with precision phi.
%                    Alpha has non-informative prior and Phi has gamma prior
%                    characterised by a degree of belief term, m0 and
%                    standard deviation term s0.
%                    $\phi \sim \rm{G}(m_0/2,m_0 s_0^2/2)$. 
%                    Samples from marginal posterior distribution of alpha.
%                    t random walk jumping distribution used.          
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Input parameters:
% ahat(n,1)          Non-linear least squares estimates of model
%                    parameters.
% fun                Function handle that takes as input alpha values
%                    and returns the non linear least squares function, f,
%                    and the matrix of derivatives of f with respect to
%                    each input, J.
% m0                 Prior degrees of freedom.
% s0                 Prior estimate of standard deviation.
% M                  Chain length for MCMC.
% N                  Number of parallel chains.
% M0                 Burn-in length for MCMC.
% Q(nQ,1)            Quantiles to be evaluated. Ranges from 0 (minimum of
%                    sample) to 100 (maximum of sample). 50 represents
%                    the median value.
% I(true or false)   Logical variable, if I = true, then a scale factor for
%                    the jumping distribution has to be input.
%                    If I = false, then the scale is automatically determined
%                    such that the acceptance probability is around 25%.
%           Note:    A scale factor is multiplied to the Cholesky factor of
%                    the variance matrix of the jumping distribution. It
%                    regulates the spread of the jumping distribution.
%                    A scale factor greater than 1 could lead to faster
%                    convergence at the price of lower acceptance
%                    probabilities. A scale factor less than 1 could lead
%                    to slower convergence, however the acceptance
%                    probability would be higher.
%                    For a Gaussian random walk jumping distribution, the
%                    recommended acceptance probability is around 25%
%                    and the scale factor is adjusted to yield this
%                    level of acceptance.
% Output parameters:
% S(2+nQ,n)          Summary of A
%                    [ mean
%                      standard deviation
%                      percentile limits ]
%                    where the percentile limits are given by Q.
% aP(N,1)            Acceptance percentages for AA calculated for each
%                    parallel chain.
% Rh(n,1)            Estimate of convergence. Theoretically Rh >= 1, and
%                    the closer to 1, the more evidence of convergence. For
%                    example, if Rh < 1.05, then the expected improvement
%                    in the estimate of the standard deviation will be no
%                    bigger than 5 %.
% Ne(n,1)            Estimate of the number of effective number of
%                    independent draws.
%           Note:    S, Rh and Ne are calculated post-burn, ie from sample
%                    M0+1 onward.
% AA(M,N,n)          Array storing the chains: A(i,j,k) is the
%                    kth element of the parameter vector stored as the
%                    ith member of the jth chain.
%                    AA(1,j,:) = A0(:,j).
% IAA(M,N)           Acceptance indices.
%                    IAA(i,j) = 1 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    accepted so that AA(i,j,:) = as.
%                    IAA(i,j) = 0 means that the proposal as(n,1)
%                    generated at the ith step of the jth chain was
%                    rejected so that AA(i,j,:) = AA(i-1,j,:), i > 1.
% sc                 Scale factor used for the computation. 
%--------------------------------------------------------------------------
% [S,aP,Rh,Ne,AA,IAA,sc] = nllsmh_trw(ahat,fun,m0,s0,M,N,M0,Q,I)
%--------------------------------------------------------------------------
% Number of parameters
  n = size(ahat,1);

% Function and gradient at ahat 
  [f,J] = fun(ahat);

% Number of data points
  m = length(f);

% sigmahat at ahat
  sigmah = norm(f)/sqrt(m-n);

% Degrees of freedom
  nu = m + m0 - n;

% Average variance term 
  s2bar = (m0*s0^2 + (m-n)*sigmah^2)/nu;
  sbar = sqrt(s2bar);
  
% QR factorisation of J
  [~,R] = qr(J,0);
  Rr = R/sbar;

% Starting point for MCMC
  e = mvtrnd(eye(n),nu,N)';
  A0 = repmat(ahat,1,N) + Rr\e;

% Target distribution 
  target = @(a)tar_trw(a,fun,m0,s0);
 
% Scale determination
  if I == true
      sc = input('scale factor');
  elseif I == false
      k = 2.4/sqrt(n);
      s = [k/4;k/2;k;2*k;4*k];
      ta = input('required acceptance probability (0-100)');
      sc = scale_trw(s,target,A0(:,1:5),Rr,nu,ta,200,5,50);
  end

% t random walk Metropolis-Hastings
  jump = @(a)jump_trw(a,Rr/sc,nu);

% Calling MCMCMH
  [S,aP,Rh,Ne,AA,IAA] = mcmcmh(target,jump,M,N,M0,Q,A0);
%--------------------------------------------------------------------------
% End of nllsmh_trw.m
%--------------------------------------------------------------------------
end