function [E,G] = eg_grw(alp,fun,tar,m0,s0)
%--------------------------------------------------------------------------
% EG_grw.m       Energy and Gradient for a specified target distribution  
%                Energy - negative log of the target distribution - tar
%                Gradient - derivative of energy with respect to each
%                input
%                Analytical gradient.
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% alp(p,1)          Parameter values - vector of alpha and log phi. 
% fun               Function handle that takes as input alpha values 
%                   and returns the non linear least squares function, f,
%                   and the matrix of derivatives of f with respect to each
%                   input, J. 
% tar               Function handle that taken as input the vector of alpha
%                   and log phi, alp, and returns the log of the target 
%                   distribution.   
% m0                Prior degrees of freedom.
% s0                Prior estimate of standard deviation. 
% 
% Outputs:
% E                 Energy, the negative log of the target density. 
% G(p,1)            Gradient, the derivative of the energy with respect to
%                   the alphas and log phi. 
%--------------------------------------------------------------------------
% [E,G] = EG_grw(alp,fun,tar,m0,s0)                                  
%--------------------------------------------------------------------------
% Number of input quantities  
  p = size(alp,1);

% Extracting alpha and phi from the input vector
  Aa = alp(1:p-1);
  Pp = exp(alp(p,:));

% Evaluating the function and Jacobian 
  [f,J] = fun(Aa);

% Length of the data vector 
  m = length(f);

% Energy 
  E = -tar(alp);

% Gradient 
  G = zeros(p,1);
  G(1:p-1) = Pp*f'*J;
  G(p) = Pp*(f'*f + m0*s0^2)/2 - (m+m0)/2;
% -------------------------------------------------------------------------
% End of EG_grw.m
% -------------------------------------------------------------------------
end 
