function [f,J] = CIR(a,X,si)
% -------------------------------------------------------------------------
% CIR.M     Function and gradient calculation for
%           least-squares circle fit in the plane.
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% a(3,1)    Parameters [x0,y0,r0]'.
% X(m,2)    Data points [(x(i),y(i))].
% si(m,1)   Standard deviation of y.
% Outputs:
% f(m,1)    Signed distances of points to circle:
%           f(i) = sqrt((x(i)-x0)^2 + (y(i)-y0)^2) - r0.
% J(m,3)    Jacobian matrix df(i)/da(j).
% -------------------------------------------------------------------------
% [f,J] = CIR(a,X,si)
% -------------------------------------------------------------------------
% Extracting data 
  x = X(:,1);
  y = X(:,2);
 
% Size of the data and number of chains 
  m = length(x);
  N = size(a,2);
 
% Making all quantities m x N so that computation can be performed in
% parallel
  A1 = repmat(a(1,:),m,1);
  A2 = repmat(a(2,:),m,1);
  A3 = repmat(a(3,:),m,1);
  X = repmat(x,1,N);
  Y = repmat(y,1,N);
  S = repmat(si,1,N);

% Location shifted data 
  xt = X - A1;
  yt = Y - A2;
  
% Radius based on the data 
  ri = sqrt(xt.*xt + yt.*yt);

% Function 
  f = (ri - A3)./S;
  if nargout == 2
      % Jacobian matrix
      J = -[xt./(ri.*S),  yt./(ri.*S), ones(length(xt), 1)./S];
  end
% -------------------------------------------------------------------------
% End of CIR.M.
% -------------------------------------------------------------------------
end 
