%% Function to calculate 95 % quantile of chi-squared distribution 
function chi_sq = calc_chi_sq_95_percent_quantile(nu)

%% 
% For values of $\nu \leq 100$, the quantiles are provided in a
% look-up table. 
% 
% For values of $\nu > 100$, the result in 
% 
% *The distribution of chi-square* 
% 
% *E B Wilson and M M Hilferty* 
% 
% *Proceedings of the National Academy of Sciences of the United States of 
% America, 17 (1931):684-688* 
% 
% that $(\chi^2/\nu)^{1/3}$ is approximately normally distributed with mean 
% $1 - 2/(9 \nu)$ and variance $2/(9 \nu)$, is used. 
% 
% In the range $\nu = 101$ to $\nu = 1000$, the approximation of Wilson and 
% Hilferty has a maximum absolute error of 0.002 5 and a maximum relative 
% percentage error of 0.002 %. 

%%
% Generate look-up table. 
table = [...
   1        3.841
   2        5.991
   3        7.815
   4        9.488
   5       11.070
   6       12.592
   7       14.067
   8       15.507
   9       16.919
  10       18.307
  11       19.675
  12       21.026
  13       22.362
  14       23.685
  15       24.996
  16       26.296
  17       27.587
  18       28.869
  19       30.144
  20       31.410
  21       32.671
  22       33.924
  23       35.172
  24       36.415
  25       37.652
  26       38.885
  27       40.113
  28       41.337
  29       42.557
  30       43.773
  31       44.985
  32       46.194
  33       47.400
  34       48.602
  35       49.802
  36       50.998
  37       52.192
  38       53.384
  39       54.572
  40       55.758
  41       56.942
  42       58.124
  43       59.304
  44       60.481
  45       61.656
  46       62.830
  47       64.001
  48       65.171
  49       66.339
  50       67.505
  51       68.669
  52       69.832
  53       70.993
  54       72.153
  55       73.311
  56       74.468
  57       75.624
  58       76.778
  59       77.931
  60       79.082
  61       80.232
  62       81.381
  63       82.529
  64       83.675
  65       84.821
  66       85.965
  67       87.108
  68       88.250
  69       89.391
  70       90.531
  71       91.670
  72       92.808
  73       93.945
  74       95.081
  75       96.217
  76       97.351
  77       98.484
  78       99.617
  79      100.749
  80      101.879
  81      103.010
  82      104.139
  83      105.267
  84      106.395
  85      107.522
  86      108.648
  87      109.773
  88      110.898
  89      112.022
  90      113.145
  91      114.268
  92      115.390
  93      116.511
  94      117.632
  95      118.752
  96      119.871
  97      120.990
  98      122.108
  99      123.225
 100      124.342... 
 ]; 

%% 
% Search for value of nu in first column of lookup table. If nu appears in
% the table, set chi_sq to be the corresponding quantile, otherwise use the
% approximation of Wilson and Hilferty. 
ind = find(nu == table(:, 1), 1); 
if ~isempty(ind)
  chi_sq = table(ind, 2); 
else 
  chi_sq = nu*((1 - 2/(9*nu) + 1.644854*sqrt(2/(9*nu)))^3); 
end 
%%
% End of calc_chi_sq_95_percent_quantile.m