%% Software to support ISO/TS 28037:2010(E)

%% Introduction
% This document runs the numerical example of Gauss-Markov regression (GMR) 
% described in *Clause 9 (Model for uncertainties and covariances 
% associated with the y_i)*. 
%
% Users of MATLAB may run the code in the corresponding M-file directly to 
% obtain the results given in ISO/TS 28037:2010(E) and may also modify the 
% data and run the code on the new data.
% 
% For users who do not have access to MATLAB, the software may be used as
% the basis for preparing implementations in other programming languages.
%
% The software is provided with a <NPL_TS_28037(2010)_MSC_L_10_001.pdf 
% software licence agreement> (REF: MSC/L/10/001) and the use of the 
% software is subject to the terms laid out in that agreement. By running 
% the M-code, the user accepts the terms of the agreement. 

close all
clear
format short 

%% Assign measurement data

%% 
% Assign x-values. 
x = [1.0 2.0 3.0 4.0 5.0 6.0 7.0 8.0 9.0 10.0]';
m = length(x);

%% 
% Assign y-values. 
y = [1.3 4.1 6.9 7.5 10.2 12.0 14.5 17.1 19.5 21.0]'; 

%%
% Assign covariance matrix associated with y-values. 
Uy = [2  1  1  1  1  0  0  0  0  0 
      1  2  1  1  1  0  0  0  0  0 
      1  1  2  1  1  0  0  0  0  0 
      1  1  1  2  1  0  0  0  0  0 
      1  1  1  1  2  0  0  0  0  0 
      0  0  0  0  0  5  4  4  4  4 
      0  0  0  0  0  4  5  4  4  4 
      0  0  0  0  0  4  4  5  4  4 
      0  0  0  0  0  4  4  4  5  4 
      0  0  0  0  0  4  4  4  4  5]; 

%% Obtain estimates of the straight line calibration function parameters and associated standard uncertainties and covariance
% Solve the Gauss-Markov regression problem to obtain best fit straight-line parameters. 

%%
% Step 1. 
Ly = chol(Uy, 'lower'); 

%% 
% Step 2. 
e = ones(m, 1);   
f = Ly\e; 
g = Ly\x; 
h = Ly\y; 

%% 
% Step 3. 
F2 = sum(f.*f); 

%% 
% Step 4. 
g0 = sum(f.*g)/F2; 
h0 = sum(f.*h)/F2; 
  
%% 
% Step 5. 
gt = g - g0*f; 
ht = h - h0*f; 
  
%% 
% Step 6. 
Gt2 = sum(gt.*gt); 
  
%% 
% Step 7. 
b = sum(gt.*ht)/Gt2; 
a = h0 - b*g0; 

%% 
% Step 8. 
u2a = 1/F2 + g0^2/Gt2;
u2b = 1/Gt2;
uab = -g0/Gt2;
  
%% 
% Step 9. 
r = ht - b*gt;
  
%% 
% Step 10. 
chi_sq_obs = sum(r.*r); 
nu = m - 2; 

%% 
% Step 11. 
chi_sq = calc_chi_sq_95_percent_quantile(nu); 

%% Display information on screen and generate figures 

%% 
% Measurement model. 
fprintf('\nMODEL FOR UNCERTAINTIES AND COVARIANCES ASSOCIATED WITH THE THE YI \n\n')
fprintf('ISO/TS 28037:2010(E) CLAUSE 9 \n') 
fprintf('EXAMPLE \n\n')

%% 
% Measurement data. 
fprintf('FITTING \n')
fprintf(['Data representing ', num2str(m),' measurement points \n'])
for i = 1:m
  fprintf('%8.1f %8.1f \n', [x(i), y(i)]) 
end 
fprintf('\n')

%% 
% Covariance matrix Uy. 
fprintf('Covariance matrix Uy \n')
disp(Uy)

%% 
% Cholesky factor Ly of Uy. 
fprintf('Cholesky factor Ly of Uy \n')
disp(Ly)

%% 
% Calculation tableau: see
% <write_gmr_tableaux.html write_gmr_tableaux.m>.
write_gmr_tableaux(f, g, h, g0, h0, gt, ht, a, b, r, '%9.4f ');

%% 
% Solution estimates. 
fprintf('Estimate of intercept \n'), fprintf('%9.4f \n\n', a)
fprintf('Estimate of slope \n'), fprintf('%9.4f \n\n', b)

%% 
% Standard uncertainties associated with solution estimates. 
fprintf('Standard uncertainty associated with estimate of intercept \n'), fprintf('%9.4f \n\n', sqrt(u2a))
fprintf('Standard uncertainty associated with estimate of slope \n'), fprintf('%9.4f \n\n', sqrt(u2b))
fprintf('Covariance associated with estimates of intercept and slope \n'), fprintf('%9.4f \n\n', uab)

%% 
% Validation of the model. 
fprintf('VALIDATION \n')
fprintf('Degrees of freedom \n'), fprintf('%4u \n\n', nu)
fprintf('Observed chi-squared value \n'), fprintf('%9.3f \n\n', chi_sq_obs)
fprintf('95 %% quantile of chi-squared distribution with %u degrees of freedom', nu), fprintf('\n%9.3f \n\n', chi_sq)
if chi_sq_obs > chi_sq 
  fprintf('CHI-SQUARED TEST FAILED - STRAIGHT-LINE MODEL IS REJECTED \n\n')
else
  fprintf('CHI-SQUARED TEST PASSED - STRAIGHT-LINE MODEL IS ACCEPTED \n\n')
end 

%% 
% Figures. 
set(0, 'DefaultLineLineWidth', 2)
set(0, 'DefaultAxesFontSize', 12)
set(0, 'DefaultAxesFontWeight', 'bold')
figure, hold on
errorbar(x, y, sqrt(diag(Uy)), 'ko', 'MarkerFaceColor', 'k', 'MarkerSize', 6)
plot(x, a + b*x, 'b-')
xlabel('\it x', 'FontName', 'Times', 'FontSize', 14)
ylabel('\it y', 'FontName', 'Times', 'FontSize', 14)
axis1 = axis; 
figure, hold on
for i = 1:m 
  plot([x(i), x(i)], [0, r(i)], 'k-')
  plot(x(i), r(i), 'ko', 'MarkerFaceColor', 'w', 'MarkerSize', 6)
end
plot([axis1(1), axis1(2)], [0, 0], 'b--')
xlabel('\it x', 'FontName', 'Times', 'FontSize', 14)
ylabel('\it r', 'FontName', 'Times', 'FontSize', 14)

%% Acknowledgements 
% The work described here was supported by the National Measurement Office
% of the UK Department of Business, Innovation and Skills as part of its
% NMS Software Support for Metrology programme. 