%% Software to support ISO/TS 28037:2010(E)

%% Introduction
% This document runs the numerical example of generalized Gauss-Markov 
% regression (GGMR) described in *Clause 10 (Model for uncertainties and 
% covariances associated with the x_i and the y_i)*. 
%
% Users of MATLAB may run the code in the corresponding M-file directly to 
% obtain the results given in ISO/TS 28037:2010(E) and may also modify the 
% data and run the code on the new data.
% 
% For users who do not have access to MATLAB, the software may be used as
% the basis for preparing implementations in other programming languages.
%
% The software is provided with a <NPL_TS_28037(2010)_MSC_L_10_001.pdf 
% software licence agreement> (REF: MSC/L/10/001) and the use of the 
% software is subject to the terms laid out in that agreement. By running 
% the M-code, the user accepts the terms of the agreement. 

close all
clear
format short 

%% Assign measurement data

%% 
% Assign x-values. 
x = [50.4 99.0 149.9 200.4 248.5 299.7 349.1]';
m = length(x);

%% 
% Assign y-values. 
y = [52.3 97.8 149.7 200.1 250.4 300.9 349.2]'; 

%%
% Assign covariance matrix associated with x-values. 
Ux = [0.5   0     0.25  0     0.25  0     0.25
      0     1.25  1     0     0     1     1
      0.25  1     1.5   0     0.25  1     1.25
      0     0     0     1.25  1     1     1
      0.25  0     0.25  1     1.5   1     1.25
      0     1     1     1     1     2.25  2
      0.25  1     1.25  1     1.25  2     2.5]; 

%%
% Assign covariance matrix associated with y-values. 
Uy = [5  1  1  1  1  1  1
      1  5  1  1  1  1  1
      1  1  5  1  1  1  1
      1  1  1  5  1  1  1
      1  1  1  1  5  1  1
      1  1  1  1  1  5  1
      1  1  1  1  1  1  5]; 

%%
% Build complete covariance matrix for x- and y-values (assuming the
% correlation associated with each (x, y)-pair is zero).
U = blkdiag(Ux, Uy);

%% Obtain estimates of the straight line calibration function parameters and associated standard uncertainties and covariance
% Solve the generalized Gauss-Markov regression problem to obtain best fit straight-line parameters. 

%%
% Step 1. Initial approximation using weighted least squares: see
% <algm_wls_steps_1_to_8.html algm_wls_steps_1_to_8.m>.
[ai, bi] = algm_wls_steps_1_to_8(x, y, sqrt(diag(Uy)));

%% 
% Round approximations to parameters in step 1 to four decimal places. 
% (This step is included to produce the results given in ISO/TS 28037:2010: 
% the step would not generally be performed.)
ai = round(10000*ai)/10000;
at{1} = ai; 
bi = round(10000*bi)/10000;
bt{1} = bi; 

%%
% Initial approximation.
tt{1} = [x; at{1}; bt{1}]; 

%% 
% Loop through steps 2 to 9 until convergence criteria are satisfied. 
% (In this example, convergence is considered to have been achieved when 
% the magnitudes of all increments are no greater than 1e-7. For general 
% user data, it is suggested that convergence is considered to have been
% achieved when the magnitudes of all increments *relative to the initial 
% approximations to the straight-line parameters* are no greater than 1e-7. 
% In this case, the tolerance can be assigned using the command 
% tol = 1e-7*norm([ai, bi]);) 

%% 
% Assign tolerances and initialize variables. 
tol = 1e-7;
dt{1} = []; f{1} = []; J{1} = []; ft{1} = []; Jt{1} = []; g{1} = []; 
H{1} = []; M{1} = []; q{1} = []; 

%% 
% Steps 2 to 9: see
% <algm_ggmr_cholesky_steps_2_to_9.html algm_ggmr_cholesky_steps_2_to_9.m>.
ind = 1;
[tt, dt, f, J, ft, Jt, g, H, M, q] ... 
  = algm_ggmr_cholesky_steps_2_to_9(x, y, U, tt, dt, f, J, ft, Jt, g, H, M, q, ind); 

while any(abs(dt{ind}) > tol) 
%% 
% Update iteration number. 
  ind = ind + 1; 

%% 
% Step 10. Repeat steps 2 to 9 until convergence has been achieved: see
% <algm_ggmr_cholesky_steps_2_to_9.html algm_ggmr_cholesky_steps_2_to_9.m>.
  [tt, dt, f, J, ft, Jt, g, H, M, q] ... 
    = algm_ggmr_cholesky_steps_2_to_9(x, y, U, tt, dt, f, J, ft, Jt, g, H, M, q, ind); 
end 
a = tt{ind+1}(m+1); 
b = tt{ind+1}(m+2); 

%% 
% Step 11. Partition Cholesky factor and evaluate uncertainties. 
M22 = M{ind}(m+1:m+2, m+1:m+2); 
m11 = M22(1, 1); 
m21 = M22(2, 1); 
m22 = M22(2, 2); 
u2a = (m22^2 + m21^2)/(m11^2*m22^2); 
u2b = 1/(m22^2); 
uab = -m21/(m11*m22^2); 

%% 
% Step 12. Form observed chi-squared value and degrees of freedom. 
chi_sq_obs = sum(ft{ind}.*ft{ind}); 
nu = m-2; 

%% 
% Step 13. Calculate 95 % quantile of chi-squared distribution: see 
% <calc_chi_sq_95_percent_quantile.html calc_chi_sq_95_percent_quantile.m>. 
chi_sq = calc_chi_sq_95_percent_quantile(nu); 

%% Display information on screen

%% 
% Measurement model. 
fprintf('\nMODEL FOR UNCERTAINTIES AND COVARIANCES ASSOCIATED WITH THE XI AND THE YI \n\n')
fprintf('ISO/TS 28037:2010(E) CLAUSE 10 \n') 
fprintf('EXAMPLE \n\n')

%% 
% Measurement data. 
fprintf('FITTING \n')
fprintf(['Data representing ', num2str(m),' measurement points \n'])
for i = 1:m
  fprintf('%8.1f %8.1f \n', [x(i), y(i)]) 
end 
fprintf('\n')

%% 
% Covariance matrix Ux. 
fprintf('Covariance matrix Ux \n')
disp(Ux)

%%
% Cholesky factor Lx of Ux. 
fprintf('Cholesky factor Lx of Ux \n')
disp(chol(Ux, 'lower'))

%% 
% Covariance matrix Uy. 
fprintf('Covariance matrix Uy \n')
disp(Uy)

%%
% Cholesky factor Ly of Uy. 
fprintf('Cholesky factor Ly of Uy \n')
disp(chol(Uy, 'lower'))

%%  
% Initial approximations to parameters. 
fprintf('Initial approximation to intercept \n'), fprintf('%9.4f \n\n', ai)
fprintf('Initial approximation to slope \n'), fprintf('%9.4f \n\n', bi)
  
%% 
% Summary of iterative procedure: see
% <write_ggmr_tableau.html write_ggmr_tableau.m>.
write_ggmr_tableau(tt, dt, ind, '%14.4e ');

%%
% Matrices M and M22 at final iteration.
fprintf('Matrix M at final iteration \n')
disp(M{ind})
fprintf('Matrix M22 at final iteration \n')
disp(M22)

%% 
% Solution estimates. 
fprintf('Estimate of intercept \n'), fprintf('%9.4f \n\n', a)
fprintf('Estimate of slope \n'), fprintf('%9.4f \n\n', b)

%% 
% Standard uncertainties associated with solution estimates. 
fprintf('Standard uncertainty associated with estimate of intercept \n'), fprintf('%9.4f \n\n', sqrt(u2a))
fprintf('Standard uncertainty associated with estimate of slope \n'), fprintf('%9.4f \n\n', sqrt(u2b))
fprintf('Covariance associated with estimates of intercept and slope \n'), fprintf('%9.4f \n\n', uab)

%% 
% Validation of the model. 
fprintf('VALIDATION \n')
fprintf('Degrees of freedom \n'), fprintf('%4u \n\n', nu)
fprintf('Observed chi-squared value \n'), fprintf('%9.3f \n\n', chi_sq_obs)
fprintf('95 %% quantile of chi-squared distribution with %u degrees of freedom', nu), fprintf('\n%9.3f \n\n', chi_sq)
if chi_sq_obs > chi_sq 
  fprintf('CHI-SQUARED TEST FAILED - STRAIGHT-LINE MODEL IS REJECTED \n\n')
else
  fprintf('CHI-SQUARED TEST PASSED - STRAIGHT-LINE MODEL IS ACCEPTED \n\n')
end 

%% Acknowledgements 
% The work described here was supported by the National Measurement Office
% of the UK Department of Business, Innovation and Skills as part of its
% NMS Software Support for Metrology programme. 