%% Additive measurement model
% The measurement model is the additive model
%
% $$Y = f(X_1, X_2, ..., X_N) = X_1 + X_2 + \cdots + X_N.$$
%
% This function evaluates the model for values of the input quantities and 
% (optionally) calculates sensitivity coefficients. The input to the 
% function is a matrix *x* of *N* rows and *M* columns containing *M* sets 
% of values of the *N* input quantities in the model. The function returns 
% in *y* the *M* values of the output quantity, with the rth value 
% formed by evaluating the model for the values of the input quantities in 
% the rth column of *x*.
%
% When *M = 1*, the function returns in *c* the *N* sensitivity 
% coefficients, which are all equal to one for this model.
%
% In an application of the GUM uncertainty framework, the function is 
% called with *M = 1* and *x* contains the estimates of the input 
% quantities. In an application of the Monte Carlo method, the function 
% is called with *M* equal to the number of Monte Carlo trials, and the 
% rth column of *x* contains random draws from the probability 
% distributions for the input quantities corresponding to the rth 
% Monte Carlo trial.
%
% See sections 3, 4.2 and 5.4.1 of M G Cox, P M Harris and I M Smith, 
% Software specifications for uncertainty evaluation, <ms7.pdf 2010>.

function [y, c] = model_additive(x)

%%
% Input

%%%
%  x     N x M     Values of the input quantities

%%
% Output

%%%
%  y     1 x M     Values of the output quantity 
%  c     1 x N     Sensitivity coefficients (for the case M = 1)

%%
% Use the size of *x* to obtain values for *N* and *M*.

   [N, M] = size(x);

%%
% Form values of the output quantity.

   if N == 1
      y = x;
   else
      y = sum(x);
   end

%%
% Form values of the sensitivity coefficients (for the case *M = 1*).

   if M == 1
      c = ones(1,N);
   end

%% Program identifier
% NPLUnc_101 Open-Source Software (Release 1.1): Software for measurement uncertainty evaluation
%
% M G Cox, P M Harris and I M Smith, Mathematics and Scientific Computing Group, 
% National Physical Laboratory, UK
%
% Crown copyright 2011